// CLDemo.cpp
//
// This program is an example of how to easily use existing software which generates RS-232 style command strings with
// the Vaunix LDA product family. This program could be used by invoking it from a script, or more likely the code within it
// can be used to integrate support for the Vaunix LDA products into a test program which was designed to send command strings
// to older hardware that relies on RS-232 style command strings.
//
//	This demo program shows how to take a command string and translate it into the commands to send to the LDA device
//	In an actual test program the initialization subroutine would be placed at the beginning of the test process, and the 
//	shutdown subroutine would be placed at the end of the test process. In between, whenever a command was to be sent to the
//	attenuator a call would be made to the LDA_Command function with the command string.
//
//	The translation from whatever style of command strings are used in the test process to Vaunix LDA commands is handled by
//	the LDA_Command function.  
//
//	For simplicity this demo program has a couple of commands implemented as examples
//		DNNNNNN	 for device to use. The number following the D is the serial number of the device. If not present the first device found is used.
//		CNN		 for the channel number to use. If not present Channel 1 is used. 
//		A?		 returns the attenuation setting
//		BNN.NN	 sets the attenuation value based on the number following the command.	 
//		QN		 If N = 1 print messages, if N = 2 to 3 turn on trace messages for debugging
//
//  Note that the unusual syntax for a command line is chosen to match the example serial commands. Spaces should separate the command line options
//
//	An additional command has been added to allow entering and sending a series of commands
//
//		I		enters interactive mode
//		X		exit the program

#include "stdafx.h"
#include <iostream>
#include "vnx_LDA_api.h"

// --------------------------- Constants --------------------------------------------
#define RESULT_MAX 20					// the maximum size of a result string
#define MAX_MSG 32						// the maximum size of error and status message strings

#define NO_TRACE 0						// the value for no trace messages



// --------------------------- Allocations -------------------------------------------
static DEVID MyDevices[MAXDEVICES];		// I have statically allocated this array for convenience
										// It holds a list of device IDs for the connected devices
										// They are stored starting at MyDevices[0]

static wchar_t errmsg[32];				// For the status->string converter
static char result[RESULT_MAX+1];		// for this example the result string is a C standard char string

static int Channel = 1;					// we just default to the first channel
static int SerialNumber = 0;			// the serial number of the selected device
static DEVID DeviceID = 0;				// the handle for the device we are working with
static int TraceLevel = 0;				// 0 for no trace messages, 1 to 3 for more detailed test and debugging messages
static int NumDevices = 0;				// the number of LDA devices that are connected
static bool HaveGoodDevice = false;		// true if we have a device that either matched the user's selected serial number, or they did not define a serial number
										// and we are using the first device available.
static float Attenuation = 0;			// default attenuation is 0db, entered as a floating point value

static bool Interactive = false;		// true if the user wants to enter multiple commands

static _TCHAR str0[16] = { 0 };			// we don't use the full command line in our parser, so this entry is unused
static _TCHAR str1[16] = { 0 };			// these will be our entered commands, parsed as tokens from the intput string
static _TCHAR str2[16] = { 0 };
static _TCHAR str3[16] = { 0 };
static _TCHAR str4[16] = { 0 };
static _TCHAR str5[16] = { 0 };
static _TCHAR str6[16] = { 0 };
static _TCHAR str7[16] = { 0 };
static _TCHAR str8[16] = { 0 };

static _TCHAR * LocalArgs[9] = { str0, str1, str2, str3, str4, str5, str6, str7, str8 };




// --------------------------- Variables ---------------------------------------------

// --------------------------- Support Routines --------------------------------------


/* A function to display an error status as a Unicode string */
wchar_t* fnLDA_perror(LVSTATUS status) {
	wcscpy_s(errmsg, MAX_MSG, L"STATUS_OK");
	if (BAD_PARAMETER == status) wcscpy_s(errmsg, MAX_MSG, L"BAD_PARAMETER");
	if (BAD_HID_IO == status) wcscpy_s(errmsg, MAX_MSG, L"BAD_HID_IO");
	if (DEVICE_NOT_READY == status) wcscpy_s(errmsg, MAX_MSG, L"DEVICE_NOT_READY");
	if (FEATURE_NOT_SUPPORTED == status) wcscpy_s(errmsg, MAX_MSG, L"FEATURE_NOT_SUPPORTED");
	if (INVALID_DEVID == status) wcscpy_s(errmsg, MAX_MSG, L"INVALID_DEVID");

	return errmsg;
}

// -- one way to check for errors --
void CheckAPISet(LVSTATUS status)
{
	if (status & 0x80000000)
	{
		wprintf(L"*** Error: LDA API returned status = %x, %s ***\n", status, fnLDA_perror(status));
	}

}

/* A function to display the status as a Unicode string */
wchar_t* fnLDA_pstatus(LVSTATUS status) {
	wcscpy_s(errmsg, MAX_MSG, L"STATUS_OK");

	// Status returns for DevStatus
	if (INVALID_DEVID == status) wcscpy_s(errmsg, MAX_MSG, L"INVALID_DEVID");
	if (DEV_CONNECTED == status) wcscpy_s(errmsg, MAX_MSG, L"DEV_CONNECTED");
	if (DEV_OPENED == status) wcscpy_s(errmsg, MAX_MSG, L"DEV_OPENED");
	if (SWP_ACTIVE == status) wcscpy_s(errmsg, MAX_MSG, L"SWP_ACTIVE");
	if (SWP_UP == status) wcscpy_s(errmsg, MAX_MSG, L"SWP_UP");
	if (SWP_REPEAT == status) wcscpy_s(errmsg, MAX_MSG, L"SWP_REPEAT");
	if (SWP_BIDIRECTIONAL == status) wcscpy_s(errmsg, MAX_MSG, L"SWP_BIDIRECTIONAL");
	if (PROFILE_ACTIVE == status) wcscpy_s(errmsg, MAX_MSG, L"PROFILE_ACTIVE");

	return errmsg;
}

int InitDLL(int tracelevel)
{
	int NumDevices;

	fnLDA_SetTestMode(FALSE);

	if (tracelevel < 0 || tracelevel > 3) tracelevel = 0;		// clip out out of range tracelevel parameters

	if (tracelevel < 3)
		fnLDA_SetTraceLevel(tracelevel, tracelevel, false);		// typical tracing for DLL code debugging
	else
		fnLDA_SetTraceLevel(tracelevel, tracelevel, true);		// enable the device detect tracing in the DLL

	NumDevices = fnLDA_GetNumDevices();

	if ((tracelevel != NO_TRACE) && (NumDevices == 0)) {
		printf("No device found\n");
	}

	if (NumDevices == 1) {
		if (tracelevel != NO_TRACE) printf("Found %d Device\n", NumDevices);
	}
	else {
		if (tracelevel != NO_TRACE) printf("Found %d Devices\n", NumDevices);
	}

	NumDevices = fnLDA_GetDevInfo(MyDevices);

	return NumDevices;
}

// this function returns the DeviceID for a specific LDA serial number
// if no device with that serial number is present it sets HaveGoodDevice false
// 
DEVID DevBySerialNumber(int NumDevices, int SerialNumber)
{
	int i;
	DEVID Device = 0;											// 0 indicates we did not find the device

	if (NumDevices < 0 || NumDevices >= MAXDEVICES) return 0;	// defend our arrays

	for (i = 0; i < NumDevices; i++)
	{
		if (SerialNumber == fnLDA_GetSerialNumber(MyDevices[i]))
		{
			if (TraceLevel != NO_TRACE) printf("Found LDA with serial number %d\n", SerialNumber);
			Device = i;
			HaveGoodDevice = true;
			break;
		}
	}

	return Device;
}


// this function translates each command into either an LDA command, or a device, mode or channel selection
// it returns a string which represents its result, or status, and the length of the string or an error code
//
// in this example the commands are from the set:
//		DNNNNNN	 for device to use. The number following the D is the serial number of the device. If not present the first device found is used.
//		CNN		 for the channel number to use. If not present Channel 1 is used. 
//		A?		 returns the attenuation setting
//		BNN.NN	 sets the attenuation value based on the number following the command.	 
//		QN		 If N = 1 print messages, if N = 2 to 3 turn on trace messages for debugging
//		I		 select interactive mode
//		X		 exit.

int LDA_Command(_TCHAR * command, char * result)
{
	int i;
	int ResultLength = 0;						// a result length of 0 indicates no result string, positive values are the number of chars in the result string
	int itemp;
	int temp;
	int AttenHR;
	float ATemp;

	// defend against a missing or null result pointer
	if (result == NULL) return 0;


	wstring thisParam(command);					// make a copy of the command string

	// look at the first character of the command to see what we should do
	if (command[0] == L'D')
	{
		thisParam = wstring(thisParam.begin() + 1, thisParam.end());		// trim off the D

		// we have a serial number for the device the user wants
		temp = _wtoi(thisParam.c_str());									// get the serial number the user wants to use

		DeviceID = DevBySerialNumber(NumDevices, temp);

		if (HaveGoodDevice == true)											// we have an LDA matching the user's serial number selection
		{
			if (TraceLevel != NO_TRACE) printf("Opening Device = %d\n", DeviceID);
		}
		else
		{
			if (TraceLevel != NO_TRACE) printf("Selected Device %d Not Found, opening first device\n", temp);
			HaveGoodDevice = true;
			DeviceID = 0;
		}

		// We need to init the device (open it) before we can do anything else ---
		itemp = fnLDA_InitDevice(MyDevices[DeviceID]);

		if (itemp) {
			printf("InitDevice returned DLL error code %x\n", itemp);
		}

		// This command does not generate a result string
		result[0] = 0;
		return 0;


	}
	else if (command[0] == L'C')
	{
		thisParam = wstring(thisParam.begin() + 1, thisParam.end());		// trim off the C

		// we have the channel for the device that the user wants to use
		temp = _wtoi(thisParam.c_str());

		// handle the case where the user does not specify a device
		if (HaveGoodDevice != true)
		{
			HaveGoodDevice = true;
			DeviceID = 0;

			// We need to init the device (open it) before we can do anything else ---
			itemp = fnLDA_InitDevice(MyDevices[DeviceID]);

			if (itemp) {
				printf("InitDevice returned DLL error code %x\n", itemp);
			}

		}

		// make sure the channel number the user entered is reasonable
		if (temp > 0 && (temp <= fnLDA_GetNumChannels(MyDevices[DeviceID])))
		{
			Channel = temp;
		}

		itemp = fnLDA_SetChannel(MyDevices[DeviceID], Channel);
		CheckAPISet(itemp);

		// This command does not generate a result string
		result[0] = 0;
		return 0;
	}
	else if (command[0] == L'A')
	{
		// check if the ? is present
		if (command[1] == L'?')
		{
			// handle the case where the user does not specify a device and this command is first
			if (HaveGoodDevice != true)
			{
				HaveGoodDevice = true;
				DeviceID = 0;

				// We need to init the device (open it) before we can do anything else ---
				itemp = fnLDA_InitDevice(MyDevices[DeviceID]);

				if (itemp) {
					printf("InitDevice returned DLL error code %x\n", itemp);
				}

			}

			// get the current attenuation for the selected channel
			AttenHR = fnLDA_GetAttenuationHR(MyDevices[DeviceID]);
			CheckAPISet(AttenHR);

			if (TraceLevel != NO_TRACE) printf("AttenHR = %x\n", AttenHR);

			if (AttenHR & 0x80000000)
			{
				// we have an error, so just return an error message
				strcpy(result, "A?_ERROR");
				return (int)strlen(result);
			}
			else
			{
				// enforce the requested format of the attenuation results
				itemp = fnLDA_GetMinAttenStepHR(MyDevices[DeviceID]);

				if (!(itemp & 0x80000000) && itemp != 0)
				{
					// divide by minimum attenuator step in .05 db units, division will round down
					AttenHR = AttenHR / itemp;
					// back to the correct scale, at minimum step resolution
					AttenHR = AttenHR * itemp;
				}

				// convert to floating point representation
				ATemp = (float) AttenHR / 20;

				// return the result as a string
				sprintf(result, "%.2f", ATemp);				// two digits after the decimal point
				return (int)strlen(result);
			}

		}


	}
	else if (command[0] == L'B')
	{
		thisParam = wstring(thisParam.begin() + 1, thisParam.end());		// trim off the B

		i = thisParam.length() - 1;		// get an index to the last character in the string

		if (thisParam[i] == L'E')
		{
			thisParam = wstring(thisParam.begin(), thisParam.end() - 1);	// trim off any trailing E
		}

		// set the attenuation to the value 
		Attenuation = (float)_wtof(thisParam.c_str());	// cast to a float, _wtof actually returns a double

		AttenHR = (int)(Attenuation * 20);				// using the High Res API with .05 db units

		// handle the case where the user does not specify a device and this command is first
		if (HaveGoodDevice != true)
		{
			HaveGoodDevice = true;
			DeviceID = 0;

			// We need to init the device (open it) before we can do anything else ---
			itemp = fnLDA_InitDevice(MyDevices[DeviceID]);

			if (itemp) {
				printf("InitDevice returned DLL error code %x\n", itemp);
			}

		}
		
		// set the attenuation for the current channel
		itemp = fnLDA_SetAttenuationHR(MyDevices[DeviceID], AttenHR);
		CheckAPISet(itemp);

		if (TraceLevel != NO_TRACE) printf("Scaled Attenuation = %d\n", AttenHR);

		// return an ACK string (or character if desired)
		// result[0] = 0x06;	// the ASCII <ack> character
		// result[1] = 0x00;	// null termination needed for the string
		// return 1;

		strcpy(result, "ACK");
		return (int)strlen(result);
	}
	else if (command[0] == L'Q')
	{
		thisParam = wstring(thisParam.begin() + 1, thisParam.end());		// trim off the Q
		temp = _wtoi(thisParam.c_str());

		if (temp >= 0 && temp < 4)
		{
			TraceLevel = temp;
			if (TraceLevel < 3)
				fnLDA_SetTraceLevel(TraceLevel, TraceLevel, false);		// typical tracing for DLL code debugging
			else
				fnLDA_SetTraceLevel(TraceLevel, TraceLevel, true);		// enable the device detect tracing in the DLL
		}

		if (TraceLevel != NO_TRACE) printf("Trace Level = %d\n", TraceLevel);

		// This command does not generate a result string
		result[0] = 0;
		return 0;

	}
	else if (command[0] == L'I')
	{
		Interactive = true;
		// This command does not generate a result string
		result[0] = 0;
		return 0;
	}
	else if (command[0] == L'X')
	{
		Interactive = false;
		// This command does not generate a result string
		result[0] = 0;
		return 0;
	}
	else
	{
		// unknown command, so return an error string
		strcpy(result, "INVALID COMMAND");
		return (int) strlen(result);
	}

	result[0] = 0;
	return 0;
}

// --------------------------- Command Line Main -------------------------------------

int _tmain(int argc, _TCHAR *argv[])
{

	int i, j;
	int itemp;
	char UserInput[102];
	char Token[102];
	int TokenStart;
	int TokenLength;
	int TokenIndex;
	

	// vars to handle our original command line or a user entered string
	int ArgCount = argc;
	bool FirstPass = true;

	// guarantee our string ends...
	UserInput[101] = 0;
	Token[101] = 0;


	// start by initializing the DLL and finding out how many LDA devices are connected
	NumDevices = InitDLL(NO_TRACE);

	// quit if we don't have any devices
	if (NumDevices == 0)
	{
		printf("No device found\n");
		// return 0;
	}

ProcessCommands:
	// process each command string in the command line
	// argv[0] is the unparsed command line, which we don't use
	for (i = 1; i < ArgCount; ++i)
	{
		// wprintf(L"command %d %s\n", i, argv[i]);
		if (FirstPass)
		{
			itemp = LDA_Command(argv[i], result);
		}
		else
		{
			if (ArgCount > 8) ArgCount = 8;					// defend our array
			itemp = LDA_Command(LocalArgs[i], result);
		}


		if (itemp > 0)
		{
			printf("%s\n", result);
		}

	}

	FirstPass = false;
	// get ready to parse interactive user input
	TokenStart = 0;
	TokenLength = 0;
	TokenIndex = 1;				// this is our token string index, it can run from 1 to 8, the max number of input tokens we accept per line

	// if the user wants to enter commands interactively we will keep asking for input until the user exits
	if (Interactive)
	{
		// get another line of input from the user
		fgets(UserInput, 100, stdin);

		// generate our array of input tokens
		for (i = 0; i < strlen(UserInput); i++)
		{
			UserInput[i] = toupper(UserInput[i]);		// lets ignore the case of the input commands
			if (TokenIndex > 7) TokenIndex = 7;			// defend our array

			if (isspace((int)UserInput[i]))				// true for white space characters
			{
				if (TokenLength != 0)					// we have a token so lets copy it to our array of user input strings
				{
					// save this token
					if (TokenLength > 15) TokenLength = 15;		// defend our strings from a token with too many characters
					strncpy(Token, &UserInput[TokenStart], TokenLength);
					*LocalArgs[TokenIndex] = *Token;			// convert from an ansi-c style char string to a _TCHAR string

					for (j = 0; j < TokenLength; j++)
					{
						LocalArgs[TokenIndex][j] = Token[j];	// convert from an ansi-c style char string to a _TCHAR string
					}
					LocalArgs[TokenIndex][TokenLength] = 0;		// stuff in an end of string marker after the token string

					// get ready to look for the next token
					TokenIndex++;								// the next command token slot in the array of command tokens
					TokenLength = 0;
					TokenStart = i + 1;
				}
				else
				{
					TokenStart = i + 1;						// we are a space, so the next place a  token could start is the next character
				}
			}
			else
			{
				// this is a non-whitespace character
				TokenLength++;
			}

		}
		// we usually have a token which ends without whitespace after it, so handle that case
		if (TokenLength != 0)
		{
			// save this token
			if (TokenLength > 15) TokenLength = 15;		// defend our strings from a token with too many characters
			strncpy(Token, &UserInput[TokenStart], TokenLength);
			
			for (i = 0; i < TokenLength; i++)
			{
				LocalArgs[TokenIndex][i] = Token[i];	// convert from an ansi-c style char string to a _TCHAR string
			}
			LocalArgs[TokenIndex][TokenLength] = 0;		// stuff in an end of string marker after the token string
			TokenIndex++;
		}

		ArgCount = TokenIndex;					// the index starts at 1 because argv[0] is traditionally the entire command line
												// but the argc count includes argv[0], so there are two values.

		// go back and parse the command string
		goto ProcessCommands;
	}



	// we are done, close our device
	if (HaveGoodDevice == true)
	{
		if (TraceLevel != NO_TRACE) printf("Closing device %d\n", DeviceID);
		fnLDA_CloseDevice(MyDevices[DeviceID]);
	}

	// exit to OS
	return 0;
}


